<?php

/**
 * This is the model class for table "{{pricing_tier}}".
 *
 * The followings are the available columns in table '{{pricing_tier}}':
 * @property integer $id
 * @property integer $group_id
 * @property string $vendor_id
 * @property string $cost_markup
 * @property string $msrp_markdown
 * @property integer $default_type
 * @property integer $status
 * @property string $date_created
 * @property string $date_updated
 *
 * The followings are the available model relations:
 * @property PricingGroup $group
 */
class PricingTier extends CActiveRecord
{
	const STATUS_ACTIVE = 1;
	const STATUS_INACTIVE = 2;
	const STATUS_DELETED = 3;

	const TYPE_MARK_UP = 1;
	const TYPE_MARK_DOWN = 2;

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return '{{pricing_tier}}';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('group_id, desc_id, vendor_id', 'required'),
			array('group_id, desc_id, vendor_id, status', 'numerical', 'integerOnly' => true),
			//array('tier_name', 'length', 'max' => 128),
			array('cost_markup, msrp_markdown', 'length', 'max' => 10),
			array('cost_markup, msrp_markdown', 'numerical', 'integerOnly' => true, 'min' => 0, 'max' => 100),
			array('date_created, date_updated', 'safe'),
			// The following rule is used by search().
			// @todo Please remove those attributes that should not be searched.
			array('id, group_id, desc_id, vendor_id, cost_markup, msrp_markdown, status, date_created, date_updated', 'safe', 'on' => 'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
			'group' => array(self::BELONGS_TO, 'PricingGroup', 'group_id'),
			'description' => array(self::BELONGS_TO, 'PricingTierDescription', 'desc_id'),
			'vendor' => array(self::BELONGS_TO, 'Vendor', 'vendor_id'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'ID',
			'group_id' => 'Group',
			'desc_id' => 'Description',
			'vendor_id' => 'Vendor',
			//'pricing_tier_wp_role_id' => 'Pricing Tier WP Role ID',
			//'tier_name' => 'Tier Name',
			'cost_markup' => 'Cost Markup',
			'msrp_markdown' => 'MSRP Markdown',
			'default_type' => 'Price Type',
			//'sort_order' => 'Sort Order',
			'status' => 'Status',
			'date_created' => 'Date Created',
			'date_updated' => 'Date Updated',
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 *
	 * Typical usecase:
	 * - Initialize the model fields with values from filter form.
	 * - Execute this method to get CActiveDataProvider instance which will filter
	 * models according to data in model fields.
	 * - Pass data provider to CGridView, CListView or any similar widget.
	 *
	 * @return CActiveDataProvider the data provider that can return the models
	 * based on the search/filter conditions.
	 */
	public function search()
	{
		// @todo Please modify the following code to remove attributes that should not be searched.

		$criteria = new CDbCriteria;

		$criteria->compare('id', $this->id);
		$criteria->compare('group_id', $this->group_id);
		$criteria->compare('vendor_id', $this->vendor_id, true);
		$criteria->compare('cost_markup', $this->cost_markup, true);
		$criteria->compare('msrp_markdown', $this->msrp_markdown, true);
		$criteria->compare('default_type', $this->default_type);
		$criteria->compare('status', $this->status);
		$criteria->compare('date_created', $this->date_created, true);
		$criteria->compare('date_updated', $this->date_updated, true);

		return new CActiveDataProvider($this, array(
			'criteria' => $criteria,
		));
	}

	public function scopes()
	{
		return array(
			'active' => array(
				'condition' => $this->tableAlias . '.status = ' . self::STATUS_ACTIVE,
			),
			'existing' => array(
				'condition' => $this->tableAlias . '.status <> ' . self::STATUS_DELETED,
			),
		);
	}

	protected function beforeSave()
	{
		if (parent::beforeSave()) {
			if ($this->isNewRecord) {
				$this->date_created = $this->date_updated = date('Y-m-d H:i:s');
			} else {
				$this->date_updated = date('Y-m-d H:i:s');
			}
			return true;
		}
	}

	public static function getPricingType()
	{
		return [
			self::TYPE_MARK_UP => 'Mark Up',
			self::TYPE_MARK_DOWN => 'Mark Down',
		];
	}

	public static function getPriceTypeLabel($type)
	{
		return self::getPricingType()[$type];
	}

	public static function getStatusList($withColor = false)
	{
		if (!$withColor) {
			return [
				self::STATUS_ACTIVE => 'Active',
				self::STATUS_INACTIVE => 'Inactive',
				self::STATUS_DELETED => 'Deleted',
			];
		} else {
			return [
				self::STATUS_ACTIVE => '<span class="label label-primary">Active</span>',
				self::STATUS_INACTIVE => '<span class="label label-default">Inactive</span>',
				self::STATUS_DELETED => '<span class="label label-warning">Deleted</span>',
			];
		}
	}

	public static function getStatusLabelWithColor($idx)
	{
		return self::getStatusList(true)[$idx];
	}

	public static function getStatusLabel($idx)
	{
		return self::getStatusList()[$idx];
	}

	/**
	 * Returns the static model of the specified AR class.
	 * Please note that you should have this exact method in all your CActiveRecord descendants!
	 * @param string $className active record class name.
	 * @return PricingTier the static model class
	 */
	public static function model($className = __CLASS__)
	{
		return parent::model($className);
	}
}
