<?php

use Automattic\WooCommerce\Client;

// bridge class for SBB
class WooCommerce
{
    protected $client;

    protected $storeUrl;
    protected $apiConsumerKey;
    protected $apiConsumerSecret;
    protected $wordpressApi = true;
    protected $version = 'wc/v2';

    public function __construct()
    {
        $this->storeUrl = Yii::app()->params['API_SBB']['storeUrl'];
        $this->apiConsumerKey = Yii::app()->params['API_SBB']['apiConsumerKey'];
        $this->apiConsumerSecret = Yii::app()->params['API_SBB']['apiConsumerSecret'];

        $this->client = new Client(
            $this->storeUrl,
            $this->apiConsumerKey,
            $this->apiConsumerSecret,
            [
                'wp_api' => $this->wordpressApi,
                'version' => $this->version,
                'query_string_auth' => true,
                'sslverify' => false,
            ]
        );
    }

    public function batchOverwriteData($productFeeds, $params = [])
    {
        if (!Yii::app()->params['API_SBB']['enable']) {
            return [];
        }
        $toOverwrite = [];

        foreach ($productFeeds as $productFeed) {
            //$res = $this->client->get('products/?sku=' . $productFeed['sku']);
            $res = $this->client->get('products', ['sku' => $productFeed['sku']]);
            $brandName = Vendor::getVendorNameLabel($productFeed['vendor_id']);
            $description = (strlen($res[0]->description) >= strlen($productFeed['description']) ? $res[0]->description : $productFeed['description']);

            $update = [
                'id' => $res[0]->id,
                'name' => $productFeed['mpn'],
                'sku' => $productFeed['sku'],
                'description' => $description,
                'regular_price' => $productFeed['msrp'],
                'sale_price' => $productFeed['sbb_price'],
                'weight' => $productFeed['weight'],
                /*'images' => [
                    [
                        'src' => Yii::app()->params['SBB_ProductImages_Url'] . $brandName . '/' . $productFeed['id'] . '.png',
                        'position' => 0,
                    ],
                ],*/
                'dimensions' => [
                    'width' => $productFeed['width'],
                    'length' => $productFeed['length'],
                    'height' => $productFeed['height']
                ],
                // 'hwp_product_gtin' => '323212312311',
                //'stock_quantity' => (int) $productFeed['quantity'],
                'in_stock' => ($productFeed['product_status'] != Product::PRODUCT_STATUS_EOL) ? true : false,
                'status' => 'publish',
                'attributes' => [
                    [
                        'name' => 'dbid',
                        'position' => 0,
                        'visible' => false,
                        'variation' => true,
                        'options' => [$productFeed['id']]
                    ],
                ],
                'meta_data' => [
                    [
                        'key' => 'hwp_product_gtin',
                        'value' => $productFeed['ean']
                    ]
                ]
            ];

            $toOverwrite[] = $update;
        }

        $result = $this->client->post('products/batch', [
            'update' => $toOverwrite
        ]);

        return $result;
    }

    public function batchInsertEditData($productFeeds, $params = [])
    {
        if (!Yii::app()->params['API_SBB']['enable']) {
            return [];
        }
        $batchDataCreate = [];
        $batchDataUpdate = [];

        foreach ($productFeeds as $productFeed) {
            if (!isset($_SESSION['vendorCategories'][$productFeed['vendor_id']])) {
                $_SESSION['vendorCategories'][$productFeed['vendor_id']] = $this->getSBBProductsCategoryId($productFeed['vendor_id']);
            }
            $vendorCategories = $_SESSION['vendorCategories'][$productFeed['vendor_id']];

            //get brand
            $brandName = Vendor::getVendorNameLabel($productFeed['vendor_id']);
            $brandNameUcWords = ucwords(strtolower($brandName));
            $brandNameUrl = str_replace(' ', '', $brandNameUcWords);

            //checking if description is for publishing
            if (!isset($productFeed['publishing'])) {
                $res = $this->client->get('products', ['sku' => $productFeed['sku']]);
                $description = (strlen($res[0]->description) >= strlen($productFeed['description']) ? $res[0]->description : $productFeed['description']);
            } else {
                $description = $productFeed['description'];
            }

            $productFrom = $productFeed['product_from'] ?: '';

            //product that original came from website
            if ($productFrom == VendorNormalized::PRODUCT_FROM_WEBSITE) {
                $batchData = [
                    'sku' => $productFeed['sku'],
                    'description' => $description,
                    'weight' => $productFeed['weight'],
                    'dimensions' => [
                        'width' => (float) $productFeed['width'],
                        'length' => (float) $productFeed['length'],
                        'height' => (float) $productFeed['height']
                    ],
                    'in_stock' => ($productFeed['product_status'] != Product::PRODUCT_STATUS_EOL) ? true : false,
                    'status' => 'publish',
                    'attributes' => [
                        [
                            'name' => 'dbid',
                            'position' => 0,
                            'visible' => false,
                            'variation' => true,
                            'options' => [(int) $productFeed['id']]
                        ],
                    ],
                    'meta_data' => [
                        [
                            'key' => 'hwp_product_gtin',
                            'value' => (string) $productFeed['ean']
                        ],
                        [
                            'key' => 'cr_gtin',
                            'value' => (string) $productFeed['ean']
                        ],
                        [
                           'key' => 'cr_mpn',
                           'value' => (string) $productFeed['mpn'],
                        ],
                        [
                            'key' => 'cr_brand',
                            'value' => (string) $brandName
                        ]
                    ]
                ];
            //product that original came from spreadsheet
            } else {
                $regular_price = $productFeed['msrp'];
                $sale_price = ($productFeed['sbb_price'] > 0 ? $productFeed['sbb_price'] : $productFeed['cost']);
                $batchData = [
                    'name' => $productFeed['product_name'],
                    'sku' => $productFeed['sku'],
                    'description' => $description,
                    'regular_price' => (float) $regular_price,
                    'sale_price' => (float) $sale_price,
                    'weight' => $productFeed['weight'],
                    'dimensions' => [
                        'width' => (float) $productFeed['width'],
                        'length' => (float) $productFeed['length'],
                        'height' => (float) $productFeed['height']
                    ],
                    'hwp_product_gtin' => $productFeed['ean'],
                    'in_stock' => ($productFeed['product_status'] != Product::PRODUCT_STATUS_EOL) ? true : false,
                    'status' => 'publish',
                    'attributes' => [
                        [
                            'name' => 'dbid',
                            'position' => 0,
                            'visible' => false,
                            'variation' => true,
                            'options' => [(int) $productFeed['id']]
                        ],
                    ],
                    'meta_data' => [
                        [
                            'key' => 'hwp_product_gtin',
                            'value' => (string) $productFeed['ean']
                        ],
                        [
                            'key' => 'cr_gtin',
                            'value' => (string) $productFeed['ean']
                        ],
                        [
                           'key' => 'cr_mpn',
                           'value' => (string) $productFeed['mpn'],
                        ],
                        [
                            'key' => 'cr_brand',
                            'value' => (string) $brandName
                        ]
                    ]
                ];
            }

            //short description checker
            if (!empty($productFeed['qb_description'])) {
                $batchData['short_description'] = $productFeed['qb_description'];
            }
            if (!empty($productFeed['platform_1_rest_id'])) {
                $batchData['id'] = $productFeed['platform_1_rest_id'];
                $batchDataUpdate[] =  $batchData;
            } else {
                $batchData['categories'] = [['id' => $vendorCategories]];
                $batchDataCreate[] =  $batchData;
            }
        }
        $result = $this->client->post('products/batch', [
            'create' => $batchDataCreate,
            'update' => $batchDataUpdate
        ]);
        return $result;
    }

    public function batchDelete($productFeeds)
    {
        if (!Yii::app()->params['API_SBB']['enable']) {
            return [];
        }
        $batchData = [];

        foreach ($productFeeds as $productFeed) {
            if (!empty($productFeed['platform_1_rest_id'])) {
                $batchData[] = $productFeed['platform_1_rest_id'];
            }
        }

        $result = $this->client->post('products/batch', ['delete' => $batchData]);

        return $result;
    }

    public function getSBBProductsCategoryId($vendorId)
    {
        if (!Yii::app()->params['API_SBB']['enable']) {
            return [];
        }
        $brandName = Vendor::getVendorNameLabel($vendorId);

        $categoryId = '';


        $sbb = new WooCommerce();

        $categories = $sbb->client->get('products/categories', ['search' => $brandName]);
        //$categories = $sbb->client->get('products/categories?search=' .  $brandname . '&consumer_key='. $this->apiConsumerKey . '&consumer_secret=' . $this->apiConsumerSecret);
        //Dev::pvx($categories);
        foreach ($categories as $categorie) {
            $categoryName = ZCommon::cleanString($categorie->name);
            $categoryName = strtolower($categoryName);

            $vendorName = ZCommon::cleanString($brandName);
            $vendorName = strtolower($vendorName);

            if ($categoryName == $vendorName) {
                $categoryId = $categorie->id;
                break;
            }
        }

        return $categoryId;
    }

    public function getProductData($id)
    {
        if (!Yii::app()->params['API_SBB']['enable']) {
            return [];
        }
        $result = $this->client->get('products/' . $id);
        return $result;
    }


    public function getProductsByCategory($vendorId, $page)
    {
        if (!Yii::app()->params['API_SBB']['enable']) {
            return [];
        }
        $categoryId = $this->getSBBProductsCategoryId($vendorId);

        $products = [];
        if (!empty($categoryId)) {
            $sbb = new WooCommerce();
            $products = $sbb->client->get('products', ['category' => $categoryId, 'per_page' => 10, 'page' => $page, 'status' => 'publish', 'in_stock' => true]);
        }

        return $products;
    }

    public static function getProductBySKU($sku)
    {
        if (!Yii::app()->params['API_SBB']['enable']) {
            return '';
        }
        $sbb = new WooCommerce();
        $res = $sbb->client->get('products', ['sku' => $sku]);

        return $res;
    }

    public static function getImageStatus($vendor, $mpn)
    {
        if (!Yii::app()->params['API_SBB']['enable']) {
            return [];
        }
        $mpn = ZCommon::cleanSpecialCharacter($mpn);
        $imageLink = Yii::app()->params['API_SBB']['storeUrl'].'/wp-content/uploads/'.rawurlencode($vendor).'/'.rawurlencode(trim($mpn)).'.jpg';
        $imageLinkHeaders = @get_headers($imageLink);
        $imageStatus = ($imageLinkHeaders && stripos($imageLinkHeaders[0], '200') ? VendorNormalized::IMAGE_STATUS_EXIST : VendorNormalized::IMAGE_STATUS_MISSING);
        $image['status'] = $imageStatus;
        $image['link'] = $imageLink;
        return $image;
    }
}
