<?php

/**
 * This is the model class for table "{{product_pricing_tier_lookup}}".
 *
 * The followings are the available columns in table '{{product_pricing_tier_lookup}}':
 * @property integer $id
 * @property integer $product_id
 * @property integer $selected_price_type
 * @property string $cost_markup_computed
 * @property string $msrp_markdown_computed
 * @property integer $pricing_tier_id
 * @property integer $group_id
 * @property integer $status
 *
 * The followings are the available model relations:
 * @property Product $product
 * @property PricingGroup $group
 * @property PricingTier $pricingTier
 */
class ProductPricingTierLookup extends CActiveRecord
{
	const STATUS_ACTIVE = 1;
	const STATUS_INACTIVE = 2;
	const STATUS_DELETED = 3;
	
	CONST WC_STATUS_PENDING = 0;
	CONST WC_STATUS_SUCCESS = 1;
	CONST WC_STATUS_ERROR = 2;
	CONST WC_STATUS_DELETED = 3;
	CONST WC_STATUS_LOCKED = 4;
	CONST WC_STATUS_DEACTIVATED = 5;
	CONST WC_STATUS_UNPUBLISHED = 6;

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return '{{product_pricing_tier_lookup}}';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('product_id, selected_price_type, cost_markup_computed, msrp_markdown_computed, pricing_tier_id, group_id', 'required'),
			array('product_id, selected_price_type, pricing_tier_id, group_id, status', 'numerical', 'integerOnly'=>true),
			array('cost_markup_computed, msrp_markdown_computed', 'length', 'max'=>10),
			// The following rule is used by search().
			// @todo Please remove those attributes that should not be searched.
			array('id, product_id, selected_price_type, cost_markup_computed, msrp_markdown_computed, pricing_tier_id, group_id, status', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
			'product' => array(self::BELONGS_TO, 'Product', 'product_id'),
			'group' => array(self::BELONGS_TO, 'PricingGroup', 'group_id'),
			'pricingTier' => array(self::BELONGS_TO, 'PricingTier', 'pricing_tier_id'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'ID',
			'product_id' => 'Product',
			'selected_price_type' => 'Selected Price Type',
			'cost_markup_computed' => 'Cost Markup Computed',
			'msrp_markdown_computed' => 'Msrp Markdown Computed',
			'pricing_tier_id' => 'Pricing Tier',
			'group_id' => 'Group',
			'status' => 'Status',
			'woocommerce_status' => 'Woocommerce Status',
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 *
	 * Typical usecase:
	 * - Initialize the model fields with values from filter form.
	 * - Execute this method to get CActiveDataProvider instance which will filter
	 * models according to data in model fields.
	 * - Pass data provider to CGridView, CListView or any similar widget.
	 *
	 * @return CActiveDataProvider the data provider that can return the models
	 * based on the search/filter conditions.
	 */
	public function search()
	{
		// @todo Please modify the following code to remove attributes that should not be searched.

		$criteria=new CDbCriteria;

		$criteria->compare('id',$this->id);
		$criteria->compare('product_id',$this->product_id);
		$criteria->compare('selected_price_type',$this->selected_price_type);
		$criteria->compare('cost_markup_computed',$this->cost_markup_computed,true);
		$criteria->compare('msrp_markdown_computed',$this->msrp_markdown_computed,true);
		$criteria->compare('pricing_tier_id',$this->pricing_tier_id);
		$criteria->compare('group_id',$this->group_id);
		$criteria->compare('status',$this->status);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}
	
	/**
	 * Returns the static model of the specified AR class.
	 * Please note that you should have this exact method in all your CActiveRecord descendants!
	 * @param string $className active record class name.
	 * @return ProductPricingTierLookup the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	public function scopes()
	{
		return array(
			'wc_pending' => array(
				'condition' => $this->tableAlias.'.woocommerce_status = '.self::WC_STATUS_PENDING,
			),
			'wc_success' => array(
				'condition' => $this->tableAlias.'.woocommerce_status = '.self::WC_STATUS_SUCCESS,
			),
			'wc_error' => array(
				'condition' => $this->tableAlias.'.woocommerce_status = '.self::WC_STATUS_ERROR,
			),
			'wc_deleted' => array(
				'condition' => $this->tableAlias.'.woocommerce_status = '.self::WC_STATUS_DELETED,
			),
			'wc_locked' => array(
				'condition' => $this->tableAlias.'.woocommerce_status = '.self::WC_STATUS_LOCKED,
			),
			'deactivated' => array(
				'condition' => $this->tableAlias.'.status = '.PricingTier::STATUS_INACTIVE,
			),
			'deleted' => array(
				'condition' => $this->tableAlias.'.status = '.PricingTier::STATUS_DELETED,
			),
		);
	}

	public function getComputedPrice()
	{
		$pricingTierModel = $this->pricingTier;
		$pricingGroupModel = $pricingTierModel->group;
		
		if(!$pricingGroupModel) throw new Exception(`Pricing Group ID : $pricingTierModel->group_id  is not existing`);

		// Check the select price type, to use the proper product price (COST || MSRP)
		if($this->selected_price_type == PricingTier::TYPE_MARK_UP)
			return $this->cost_markup_computed;
		elseif ($this->selected_price_type == PricingTier::TYPE_MARK_DOWN)
			return $this->msrp_markdown_computed;
		else
			throw new Exception(`Selected price type is not valid: $pricingGroupModel->selected_price_type`);
	}

	public function setPriceLocked()
	{
		$this->woocommerce_status = self::WC_STATUS_LOCKED;
		return $this->save(false);
	}

	public function buildWcApiDetails()
	{
		$productModel = $this->product;
		$pricingTierModel = $this->pricingTier;
		$pricingTierDescriptionModel = $pricingTierModel->description;
		$pricingTierWpRoleModel = $pricingTierDescriptionModel->pricingTierWpRole;
		//Dev::pvx($productModel);
		if(!$productModel) throw new Exception(`Product ID : $this->product_id  is not existing`);
		if(!$pricingTierModel) throw new Exception(`Pricing Tier ID : $productModel->pricing_tier_id  is not existing`);
		if(!$pricingTierDescriptionModel) throw new Exception(`Pricing Tier ID : $productTierModel->desc_id  is not existing`);
		if(!$pricingTierWpRoleModel) throw new Exception(`Pricing Tier WP Role ID : $pricingTierDescriptionModel->pricing_tier_wp_role_id  is not existing`);

		$productDetails = array();
		$productDetails['product_id'] = trim($productModel->platform_2_rest_id);
		$productDetails['role'] = $pricingTierWpRoleModel->key;
		$productDetails['min_qty'] = 1;
		$productDetails['discount_type'] = 'flat';
		// Get the product tier to know what pricing type to push
		$productDetails['rsp_price'] = floatval($this->getComputedPrice());
		//Dev::pvx(implode('|',$productDetails));
		$productDetails['hash'] = md5(implode('|',$productDetails));

		return $productDetails;
	}

	public function setWcStatus($apiResponse, $customSuccessStatus = null)
	{
		$wcStatus = ProductPricingTierLookup::WC_STATUS_ERROR;

		// Set proper woocommerce status
		if( !empty($apiResponse) )
		{
			$wcStatus = ProductPricingTierLookup::WC_STATUS_SUCCESS;

			// Group mapping already exists. it means that the record is the same so it will still be success.
			if(property_exists($apiResponse, 'failed') && reset($apiResponse->{'failed'}) != 'Group mapping already exists.') 
				$wcStatus = ProductPricingTierLookup::WC_STATUS_ERROR;
		}

		$this->woocommerce_status = $customSuccessStatus?:$wcStatus;

		return $this->save(false);
	}

}
