<?php

/**
 * This is the model class for table "{{pricing_group}}".
 *
 * The followings are the available columns in table '{{pricing_group}}':
 * @property integer $id
 * @property string $group_name
 * @property integer $status
 * @property string $date_created
 * @property string $date_updated
 *
 * The followings are the available model relations:
 * @property PricingTier[] $pricingTiers
 */
class PricingGroup extends CActiveRecord
{
	const STATUS_ACTIVE = 1;
	const STATUS_INACTIVE = 2;
	const STATUS_DELETED = 3;

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return '{{pricing_group}}';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('group_name', 'required'),
			array('group_name', 'length', 'max' => 128),
			array('date_created, date_updated', 'safe'),
			// The following rule is used by search().
			// @todo Please remove those attributes that should not be searched.
			array('id, group_name, status, date_created, date_updated', 'safe', 'on' => 'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
			'pricingTiersDescription' => array(self::HAS_MANY, 'pricingTierDescription', 'group_id'),
			'pricingTiers' => array(self::HAS_MANY, 'pricingTier', 'group_id'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'ID',
			'group_name' => 'Group Name',
			'status' => 'Status',
			'date_created' => 'Date Created',
			'date_updated' => 'Date Updated',
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 *
	 * Typical usecase:
	 * - Initialize the model fields with values from filter form.
	 * - Execute this method to get CActiveDataProvider instance which will filter
	 * models according to data in model fields.
	 * - Pass data provider to CGridView, CListView or any similar widget.
	 *
	 * @return CActiveDataProvider the data provider that can return the models
	 * based on the search/filter conditions.
	 */
	public function search()
	{
		// @todo Please modify the following code to remove attributes that should not be searched.

		$criteria = new CDbCriteria;

		$criteria->compare('id', $this->id);
		$criteria->compare('group_name', $this->group_name, true);
		$criteria->compare('status', $this->status);
		$criteria->compare('date_created', $this->date_created, true);
		$criteria->compare('date_updated', $this->date_updated, true);

		return new CActiveDataProvider($this, array(
			'criteria' => $criteria,
		));
	}

	public function scopes()
	{
		return array(
			'active' => array(
				'condition' => $this->tableAlias . '.status = ' . self::STATUS_ACTIVE,
			),
			'existing' => array(
				'condition' => $this->tableAlias . '.status <> ' . self::STATUS_DELETED,
			),
		);
	}

	protected function beforeSave()
	{
		if (parent::beforeSave()) {
			if ($this->isNewRecord) {
				$this->date_created = $this->date_updated = date('Y-m-d H:i:s');
			} else {
				$this->date_updated = date('Y-m-d H:i:s');
			}
			return true;
		}
	}

	/*public static function getPricingGroupType()
	{
		return [
			self::TYPE_MARK_UP => 'Mark Up',
			self::TYPE_MARK_DOWN => 'Mark Down',
		];
	}*/

	public static function getPriceTypeLabel($type)
	{
		return self::getPricingGroupType()[$type];
	}

	public static function getStatusList($withColor = false)
	{
		if (!$withColor) {
			return [
				self::STATUS_ACTIVE => 'Active',
				self::STATUS_INACTIVE => 'Inactive',
				self::STATUS_DELETED => 'Deleted',
			];
		} else {
			return [
				self::STATUS_ACTIVE => '<span class="label label-primary">Active</span>',
				self::STATUS_INACTIVE => '<span class="label label-default">Inactive</span>',
				self::STATUS_DELETED => '<span class="label label-warning">Deleted</span>',
			];
		}
	}

	public static function getStatusLabelWithColor($idx)
	{
		return self::getStatusList(true)[$idx];
	}

	public static function getStatusLabel($idx)
	{
		return self::getStatusList()[$idx];
	}

	public function getTierCount()
	{
		return PricingTierDescription::model()->existing()->countByAttributes(array(
			'group_id' => $this->id
		));
	}

	/**
	 * Returns the static model of the specified AR class.
	 * Please note that you should have this exact method in all your CActiveRecord descendants!
	 * @param string $className active record class name.
	 * @return PricingGroup the static model class
	 */
	public static function model($className = __CLASS__)
	{
		return parent::model($className);
	}

	public static function getDataList() {
		$pricingGroupData = self::model()->active()->findAll(['order' => 'id DESC']);
	
		$pricingGroupList = CHtml::listData($pricingGroupData, 'id', 'group_name');

		return $pricingGroupList;
	}

	public function getPrices() {
		$pricingTier = PricingTier::model()->findAll([
			'condition' => 'group_id = :group_id and status = :status',
			'params' => [
				':group_id' => $this->id,
				':status' => PricingTier::STATUS_ACTIVE, 
			]
		]);
		
		return $pricingTier;
	}

	public static function getGroupById($id) {
		$data  = self::model()->findByPk($id);

		return $data;
	}

	public static function getProductPricingGroup($productId) {
		$productPricingTier = ProductPricingTierLookup::model()->find([
			'condition' => 'product_id = :product_id AND status IN ('. self::STATUS_ACTIVE .')',
			'params' => ['product_id' => $productId]
		]);
		
		if ($productPricingTier) { 
			$groupData = self::getGroupById($productPricingTier->group_id);
			return $groupData;
		} else {
			return null;
		}
	}
}
