<?php

/**
 * This is the model class for table "account".
 *
 * The followings are the available columns in table 'account':
 * @property integer $id
 * @property string $username
 * @property string $password
 * @property string $salt
 * @property integer $account_type
 *
 * The followings are the available model relations:
 * @property Employee[] $employees
 */
class Account extends CActiveRecord
{
	CONST STATUS_ACTIVE = 1;
	CONST STATUS_INACTIVE = 2;
	CONST STATUS_DELETED = 3;
	
	CONST TYPE_ADMIN = 1;
	CONST TYPE_SALES = 3;
	
	public $username;
	public $password;
	public $confirm_password;
	public $rememberMe;
	public $current_password;
	public $new_password;

	private $_identity;
	/**
	 * Returns the static model of the specified AR class.
	 * @return Account the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return '{{account}}';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('username, status, email_address, account_type', 'required'),
			array('account_type', 'numerical', 'integerOnly'=>true),
			array('username, password, salt', 'length', 'max'=>120),
			array('username', 'validateUserCredentials'),
			// The following rule is used by search().
			// Please remove those attributes that should not be searched.
			array('id, username, password, salt, account_type', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
			'employees' => array(self::HAS_ONE, 'Employee', 'account_id'),
			'adminAccount' => array(self::HAS_ONE, 'AdminAccount', 'account_id'),
			'userInfo' => array(self::HAS_ONE, 'UserInfo', 'user_id'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'ID',
			'username' => 'Username',
			'password' => 'Password',
			'salt' => 'Salt',
			'account_type' => 'Account Type',
			'email_address' => 'Email Address',
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		// Warning: Please modify the following code to remove attributes that
		// should not be searched.

		$criteria=new CDbCriteria;

		$criteria->compare('id',$this->id);
		$criteria->compare('username',$this->username,true);
		$criteria->compare('password',$this->password,true);
		$criteria->compare('salt',$this->salt,true);
		$criteria->compare('account_type',$this->account_type);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}
	
	public function scopes()
	{
		return array(
			'active' => array(
				'condition' => $this->tableAlias.'.status = '.self::STATUS_ACTIVE,
			),
			'existing' => array(
				'condition' => $this->tableAlias.'.status <> '.self::STATUS_DELETED,
			),
			'admins' => array(
				'condition' => $this->tableAlias.'.account_type = '.self::TYPE_ADMIN,
			),
		);
	}
	
	public function validatePassword($password)
	{
		return true;
		// echo $this->password;
		// echo '<Br /> ('.$password.') ('.$this->username.') <br />';
		// echo $this->hashPassword($password,$this->username,$this->salt);
		// exit;
		return $this->hashPassword($password,$this->username,$this->salt) === $this->password;
	}
	
	public function hashPassword($password,$username,$salt = '')
	{
		return sha1($username.$password);
	}
	
	public function getAccountList()
	{
		return self::model()->existing()->findAll();
	}
	
	protected function beforeSave(){
		if (parent::beforeSave())
		{
			if ($this->isNewRecord)
			{
				$this->date_created = $this->date_updated = date('Y-m-d H:i:s');
			}
			else{
				$this->date_updated = date('Y-m-d H:i:s');
			}
			return true;
		}
	}
	
	public static function getStatusList()
	{
		return array(
			self::STATUS_ACTIVE => 'Active',
			self::STATUS_INACTIVE => 'Inactive',
			self::STATUS_DELETED => 'Deleted',
		);
	}
	
	public static function getStatusLabel ( $status )
	{
		$statusList = self::getStatusList();
		return $statusList[$status];
	}
	
	public static function getTypeList()
	{
		return array(
			self::TYPE_ADMIN => 'Admin',
			self::TYPE_SALES => 'Sales',
		);
	}
	
	public static function getTypeLabel ( $type )
	{
		$typeList = self::getTypeList();
		return $typeList[$type];
	}
	
	public function validateUserCredentials()
	{
		$errorCount = 0;
		
		$accountModel = Account::model()->find(array(
			'condition' => 'username = :username AND status = :status AND status != '.self::STATUS_DELETED,
			'params' => array(':username' => $this->username, ':status' => $this->status)
		));
		
		if(!empty($this->username) AND $accountModel AND $accountModel->id !== $this->id)
		{
			$this->addError('username', 'Email Adress "'.$this->username.'" already exist!');
			$errorCount++;
		}
		
		if ( !empty($this->password) OR $this->isNewRecord )
		{
			if($this->isNewRecord)
			{
				$salt = time();
				$this->password = $this->hashPassword($this->password, $this->username, $salt);
				$this->confirm_password = $this->hashPassword($this->confirm_password, $this->username, $salt);
			}
			else
			{
				if ( empty($this->password) )
				{
					$this->addError('password', 'Password is required');
				}
			} 

			if ( empty($this->new_password) )
			{	
				$this->addError('password', 'New password cannot be blank.');
			}

			if ( $this->hashPassword($this->current_password,$this->username) !== $this->password )
			{	
				$this->addError('password', 'Current password didn\'t matched');
			}
		}
		else	//user doesnot want to change password so retain what ever password they have
		{
			$this->password = $this->current_password;
		}
		
		if ( $errorCount == 0 )
			return true;
		else
			return false;
	}

	public function getFullName()
	{
		return $this->userInfo->first_name.' '.$this->userInfo->last_name;
	}

	public static function getActiveAccountList()
	{
		return self::model()->active()->findAll();
	}
	
	public function isAdmin()
	{
		if( $this->account_type == self::TYPE_ADMIN )
			return true;
		return false;
	}
	
	
}
